import Layout from "@/app/components/layout";
import {
    CheckCircleIcon, ArrowPathIcon, ShieldCheckIcon, UserGroupIcon, EyeIcon, FlagIcon,
} from "@heroicons/react/24/solid";
import { FiDownload } from "react-icons/fi";
import { getAllStagesWithJoins } from "@/lib/services/stageMovementsService";
import BackButton from "@/app/components/BackButton";
import { decryptNumber } from "@/lib/encryption";
// Map stage codes to display names
const stageCodeToName = {
    FC: "First Check",
    FCR: "First Check Resubmit",
    FCQC: "First Check QC",
    FCRQC: "First Check Resubmit QC",
    EDA: "Editor assignment",
    PR: "Peer Review",
    MD: "Make Decision",
};

// Map stage display name to icon component
const stageIcons = {
    "First Check": <CheckCircleIcon className="w-6 h-6 text-green-600" />,
    "First Check Resubmit": <ArrowPathIcon className="w-6 h-6 text-orange-500" />,
    "First Check QC": <ShieldCheckIcon className="w-6 h-6 text-teal-600" />,
    "First Check Resubmit QC": <UserGroupIcon className="w-6 h-6 text-purple-600" />,
    "Editor assignment": <UserGroupIcon className="w-6 h-6 text-purple-600" />,
    "Peer Review": <EyeIcon className="w-6 h-6 text-yellow-500" />,
    "Make Decision": <FlagIcon className="w-6 h-6 text-red-600" />,
};

const stausCodeToName = {
    1 : "Completed",
    2 : "UnSubmit",
    3 : "Accept",
    4 : "Reject",
    5 : "Revise",
};

function StageMovementsCards({ movements }) {
    // const FILE_BASE_URL = "https://localhost:3000/public/QC_Files/";
    const FILE_BASE_URL =  `${process.env.NEXTAUTH_URL}/public/QC_Files/`;

    return (
        <div>
            {movements.map((move) => {
                const fromStageName = stageCodeToName[move.stage_from] || move.stage_from;
                const toStageName = stageCodeToName[move.stage_to] || move.stage_to;
                const icon = stageIcons[toStageName] || toStageName.charAt(0);
                const statusName  = stausCodeToName[move.stage_status] || move.stage_status;

                const hasFile = move.stage_uploadFile && move.stage_uploadFile.trim() !== "";
                let folder_name = "QCFiles";
                if (fromStageName === "First Check" || fromStageName === "First Check Resubmit") {
                    folder_name = "First_check";
                }
                return (
                    <div
                        key={move.stage_pid}
                        className="relative flex items-center justify-between mb-6 bg-white border border-purple-100 rounded-2xl shadow-lg px-5 py-3 hover:scale-105 transition-transform duration-300 max-h-28"
                    >
                        {/* Icon container */}
                        <div className="flex-shrink-0 flex items-center justify-center w-10 h-10 bg-purple-100 rounded-full shadow-lg text-purple-900 font-semibold text-lg">
                            {icon}
                        </div>

                        <div className="flex-1 min-w-0 ml-4">
                            <div className="flex items-center space-x-3 mb-1">
                                <span className="inline-block py-0.5 px-2 bg-purple-100 text-purple-700 rounded-sm font-semibold text-xs whitespace-nowrap">
                                    {toStageName}
                                </span>
                                <span className="text-gray-400 text-xs whitespace-nowrap">
                                    {new Date(move.stage_dispachDate).toLocaleDateString()}
                                </span>
                            </div>
                            <div className="text-sm font-semibold text-purple-900 truncate">{move.user.user_name}</div>
                            <div className="flex flex-wrap gap-x-2 gap-y-1 text-xs text-gray-700 mt-0.5">
                                <span className="font-medium">From:</span>
                                <span>{fromStageName}</span>
                                <span>→</span>
                                <span className="font-medium">To:</span>
                                <span>{toStageName}</span>
                            </div>
                            <div className="text-gray-500 italic text-xs mt-1 truncate"><span className="font-medium">Status : </span>{statusName}</div>
                            {move.stage_notes && (
                                <div className="text-gray-500 italic text-xs mt-1 truncate"><span className="font-medium">Comments : </span>{move.stage_notes}</div>
                            )}
                        </div>

                        {hasFile && (
                             
                            <a
                                 href={`/${folder_name}/${move.stage_uploadFile}`}
                                download={move.stage_uploadFile}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="ml-4 flex items-center gap-1 text-white bg-purple-600 hover:bg-purple-700 px-3 py-1 rounded text-xs font-medium select-none"
                                title="Download file"
                            >
                                <FiDownload className="w-4 h-4" /> Download
                            </a>


                        )}
                    </div>


                );
            })}
        </div>
    );
}

 export default async function Page({ params }) {
    //  const decryptedId = decryptNumber(params.id);
    //  const chapId = decryptedId;

    const { id } = await params;

    const decryptedId = decryptNumber(id);
    const chapId = decryptedId;

    // Fetch movements data server-side
    const chapDetails = await getAllStagesWithJoins(chapId);

    // Extract chapter creation date from first element
    const chapterCreateDateRaw =
        chapDetails?.[0]?.chapter?.chap_createdAt || chapDetails?.[0]?.chapter?.createdAt;

    const chapterCreateDate = chapterCreateDateRaw
        ? new Date(chapterCreateDateRaw).toLocaleDateString(undefined, {
            year: "numeric",
            month: "long",
            day: "numeric",
        })
        : null;

    return (
        <Layout>
            <div className="max-w-full px-4 py-4 mx-auto rounded-lg  space-y-4 ">

                <header>
                    <div className="flex justify-between items-center border-b border-purple-500">
                        <div className="mb-1">
                            <h2 className="text-2xl font-semibold text-gray-800">Stage Movements</h2>
                            <p className="text-gray-500 mt-1 text-sm">
                                Chapter Created on {chapterCreateDate}
                            </p>
                        </div>
                        <BackButton />
                    </div>
                </header>

                <div className="relative max-w-5xl mx-auto cursor-pointer">
                    <div className="absolute left-12 top-0 h-full w-1 bg-gradient-to-b from-purple-400 to-purple-200 rounded-full pointer-events-none" />
                    <StageMovementsCards movements={chapDetails} />
                </div>

            </div>
        </Layout>
    );
}
